#coding=utf-8

import os
import sys
import time
from threading import Thread
import json
import hashlib # 雜湊演算法，用於縮短序列化的字串。
import codecs # 開啟 utf-8 檔案
import wx
import globalPluginHandler
import gui # NVDA 的 GUI
import ui
import speech
import braille
import globalVars

class GlobalPlugin(globalPluginHandler.GlobalPlugin):
	scriptCategory = u'物件命名工具'
	def __init__(self, *args, **kwargs):
		# 初始化並載入已存替代文字
		super(GlobalPlugin, self).__init__(*args, **kwargs)
		# tags 為一字典，存放使用者設定的替代文字以及用於辨認元件的屬性。
		self.tags = None
		# 附加元件所在路徑
		self.path = globalVars.appArgs.configPath
		
		try:
			# 嘗試讀取以儲存的元件屬性與替代文字。
			file = codecs.open(self.path + '\\tags.json', encoding='utf-8')
			self.tags = json.load(file)
			file.close()
		except:
			pass

		if not isinstance(self.tags, dict):
			self.tags = dict()
		# 存放焦點事件所取得的物件，但後面沒用到。
		self.object = None
		# tag 結構為一字典，包含 attributes 為用於辨認元件的屬性集合， alt 為使用者設定的替代文字。
		self.tag = None
		# 在 NVDA -> 工具 功能表，建立替代文字功能。
		self.init_NVDA_menu()
		# 取得物件資訊的執行續羽停止標誌
		self.processThread = None
		self.stopThread = False
	
	def init_NVDA_menu(self):
		# 取得 NVDA 通知區域圖是
		self.tray = gui.mainFrame.sysTrayIcon
		# 取得 NVDA -> 工具 功能表
		self.tools_menu = self.tray.toolsMenu
		# 建立並加入替代文字功能表。
		self.alt_text_menu = wx.Menu()
		self.menu = self.tools_menu.AppendSubMenu(self.alt_text_menu, '物件命名工具 (&O)')
		# 加入替代文字內的功能表項目。
		self.item_import = self.alt_text_menu.Append(wx.ID_ANY, '匯入物件資訊 (&I)')
		self.item_export = self.alt_text_menu.Append(wx.ID_ANY, '匯出物件資訊 (&E)')
		# 綁定對應處理事件的函數。
		self.tray.Bind(wx.EVT_MENU, self.on_import, self.item_import)
		self.tray.Bind(wx.EVT_MENU, self.on_export, self.item_export)
	
	def on_import(self, evt):
		file = wx.FileSelector('匯入物件資訊', wildcard='替代文字檔 (*.json)|*.json')
		if not file:
			return None
		try:
			self.import_alt_text(file)
			wx.MessageBox(u'成功匯入 ' + file, '匯入完成')
		except Exception as e:
			wx.MessageBox(str(e), '異常', wx.ICON_ERROR)
		
	
	def on_export(self, evt):
		file = wx.SaveFileSelector('匯出物件資訊', 'json', 'alt_text')
		if not file:
			return None
		try:
			self.export_alt_text(file)
			wx.MessageBox(u'已將替代文字資訊儲存於 ' + file, '匯出成功')
		except Exception as e:
			wx.MessageBox(str(e), '異常', wx.ICON_ERROR)
		
	
	def import_alt_text(self, filename):
		with codecs.open(filename, encoding='utf-8') as file:
			tags = json.load(file)
		
		# 更新 tags
		self.tags.update(tags)
		return True
	
	def export_alt_text(self, filename):
		with codecs.open(filename, 'w', encoding='utf-8') as file:
			json.dump(self.tags, file, ensure_ascii=False)
		
		return True
	
	def terminate(self):
		# 關閉 NVDA 時儲存所有辨認屬性與替代文字，也就是儲存 tags 到檔案。
		file = codecs.open(self.path + '\\tags.json', 'w', encoding='utf-8')
		json.dump(self.tags, file, ensure_ascii=False)
		file.close()
		# 刪除替代文字功能表。
		self.tools_menu.Remove(self.menu.Id)
		self.alt_text_menu.destroy()

	def script_setAlt(self, gesture):
		# 初始化輸入替代文字的對話框
		self.dialog = Dialog()
		self.dialog.ok.Bind(wx.EVT_BUTTON, self.on_ok)
		self.dialog.cancel.Bind(wx.EVT_BUTTON, self.on_cancel)
		self.dialog.edit.Value = self.tag['alt']
		self.dialog.edit.SetFocus()
		# 讓 NVDA 來監控並顯示該對話框，能夠強制讓對話框顯示於前景，必須將對話框的 parent 設定為 gui.mainFrame
		gui.runScriptModalDialog(self.dialog)
	script_setAlt.__doc__ = u'設定提示文字'

	def on_ok(self, evt):
		# 按下確定按鈕
		evt.Skip()
		self.dialog.Hide()
		self.set_all()

	def on_cancel(self, evt):
		# 按下取消按鈕
		evt.Skip()
		self.dialog.Hide()

	def set_all(self):
		# 新增、修改或刪除 alt 到 tags
		alt = self.dialog.edit.Value
		if alt:
			if not self.tag['alt']:
				# 未設定此元件的替代文字，所以加入 tags
				self.tags[self.tag['key']] = self.tag
			# 注意縮排，不論有沒有設定過替代文字，都會修改為最新輸入的替代文字。
			self.tag['alt'] = alt
			self.object.name = alt
		else:
			# 替代文字為空，從 tags 當中刪除替代文字。
			del self.tags[self.tag['key']]
			self.tag['alt'] = ''
		
	
	def get_attributes(self, obj):
		# 取出所有辨認屬性集合
		attributes = list()
		o = obj
		while o:
			if self.stopThread:
				sys.exit()
			# attrib 為屬性字典，屬性名稱與屬性值相對應。
			attrib = dict()
			if o == obj:
				# 若非導航器所在物件，則不取其 name.
				attrib['name'] = o.name
			attrib['role'] = o.role
			attrib['description'] = o.description
			n = str(o).find(' at ')
			attrib['class'] = str(o)[:n]
			attrib['appModule.appName'] = o.appModule.appName
			try:
				attrib['indexInParent'] = o.indexInParent
			except:
				attrib['indexInParent'] = None
			'''
			發現拖累速度的原因，不知為何任何物件娶她的 children 速度非常慢，所以放棄從上層取 children 並找出該物件所引的方式，並將其加入此區塊註解。
			try:
				attrib['list_index'] = o.parent.children.index(o)
			except:
				attrib['list_index'] = -1
			'''
			# IAccessible 系列
			try:
				attrib['IAccessibleIdentity'] = o.IAccessibleIdentity.copy()
				del attrib['IAccessibleIdentity']['windowHandle']
			except:
				attrib['IAccessibleIdentity'] = None
			
			# IA2Web 系列
			try:
				attrib['IA2id'] = o.IA2Attributes.get('id')
				attrib['IA2class'] = o.IA2Attributes.get('class')
				attrib['IA2tag'] = o.IA2Attributes.get('tag')
			except:
				pass
			
			attributes.append(attrib)
			
			# 一直向上取道 parent 的盡頭。
			o = o.parent
		return attributes
	
	def event_becomeNavigatorObject(self, obj, callToSkipEvent, isFocus):
		callToSkipEvent()
		# 停止處理物件資訊的 thread
		self.stopThread = True
		if self.processThread is not None:
			self.processThread.join()
		# 啟動取得物件資訊的執行續
		self.stopThread = False
		self.processThread = Thread(target=GlobalPlugin.processObjectInfo, args=(self, obj))
		self.processThread.start()
		
	
	def processObjectInfo(self, obj):
		# 紀錄該元件的資訊並檢查是否有對應替代文字，若有責修改其 name 並讓 NVDA 進行回饋。
		if obj.appModule.appName == 'nvda':
			# 跳過與 NVDA 相關的事件
			sys.exit()
		time.sleep(0.1)
		self.object = obj
		self.tag = None
		# 取的該元件的屬性集合
		attributes = self.get_attributes(obj)
		# 根據 Python 版本，取得將字典轉換成字串的函數
		if sys.version_info.major <= 2:
			cstr = unicode
		else:
			cstr = str
		
		# 產生元件的唯一值
		key = sha(cstr(attributes))
		# 透過 key 在 tags 當中尋找
		self.tag = self.tags.get(key)
		if self.tag:
			# 有找到就設定物件的 name 為替代文字。
			obj.name = self.tag['alt']
			if self.stopThread:
				sys.exit()
			# 讓 NVDA 回饋替代文字
			braille.handler.handleUpdate(obj)
			speech.cancelSpeech()
			speech.speakObject(obj)
		else:
			# 沒找到就設定一個包含當前元件屬性，但 alt 為空的 tag, 且將 key 宜並加入
			self.tag = {'attributes': attributes, 'alt': '', 'key': key}
		
	
	def script_speakTag(self, gesture):
		if not self.tag.get('alt'):
			ui.message(u'此元件沒有替代文字')
		else:
			ui.message(self.tag.get('alt'))
		
	script_speakTag.__doc__ = u'說出當前元件的替代文字'
	__gestures = {
		'kb:NVDA+i': 'setAlt',
		'kb:NVDA+o': 'speakTag',
	}

class Dialog(wx.Dialog):
	# 設定替代文字的對話框
	def __init__(self):
		# NVDA 的 gui.mainFrame 包含 NVDA 功能表與設定視窗等，將對話框的 parent 設為 NVDA 的主視窗，能夠讓 NVDA 在適當時機控制他。
		super(Dialog, self).__init__(parent=gui.mainFrame, title='替代文字')
		# 對話框的直向 sizer
		self.sizer = wx.BoxSizer(wx.VERTICAL)
		# 建立控件並將其加入 sizer
		self.label = wx.StaticText(self, label='請輸入替代文字：')
		self.sizer.Add(self.label, wx.SizerFlags(0).Border(wx.TOP, 2))
		self.edit = wx.TextCtrl(self)
		self.sizer.Add(self.edit, wx.SizerFlags(1).Expand().Border(wx.ALL, 6))
		# 建立包含確定與取消兩個按鈕的 sizer.
		# 之前自己建立確定與取消的按鈕，不知為何，設定 parent 為 NVDA 的主視窗之後，再輸入替代文字的編輯區直接按 enter 就不會執行確定按鈕了，而使用待按鈕的 sizer 不知為何就可以。
		self.buttons = self.CreateButtonSizer(wx.OK | wx.CANCEL)
		# 透過 ID 找出確定與取消兩個按鈕的物件，類似 JS 的 document.getElementById
		self.ok = wx.FindWindowById(wx.ID_OK)
		self.cancel = wx.FindWindowById(wx.ID_CANCEL)
		# 將這個按鈕群的 sizer 加入主要的 sizer 並指定底部邊框。
		self.sizer.Add(self.buttons, flag=wx.BOTTOM, border=2)
		# 將 sizer 加入主視窗並調整適合大小。
		self.SetSizerAndFit(self.sizer)
	

def sha(string):
	# 取得字串的 hash 值。
	sha = hashlib.sha256()
	sha.update(string.encode('utf-8'))
	return sha.hexdigest()
